<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Save;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Models\Post\Save\PostSaveModel;
use Erlage\Photogram\Data\Models\Post\Save\PostSaveBuilder;

final class PostSaveContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function load(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $offset = self::$request -> findKeyOffset(PostSaveTable::ID, PostSaveTable::TABLE_NAME);

            // optional
            $collectionIdFromReq = self::$request -> findKey(
                PostSaveTable::SAVED_TO_COLLECTION_ID,
                RequestTable::PAYLOAD,
                PostSaveTable::TABLE_NAME
            );

            $isLoadingForCollection
                = self::isAvailable($collectionIdFromReq) && $collectionIdFromReq != PostSaveBuilder::defaultValues()[PostSaveTable::SAVED_TO_COLLECTION_ID];

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $postSaveTableQuery = (new Query()) -> from(PostSaveTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | select where user id is matched
            |--------------------------------------------------------------------------
            */

            $postSaveTableQuery -> where(PostSaveTable::SAVED_BY_USER_ID, self::$authedUserModel -> getId());

            /*
            |--------------------------------------------------------------------------
            | if loading for a collection, add predicate for collection id
            |--------------------------------------------------------------------------
            */

            if ($isLoadingForCollection)
            {
                $postSaveTableQuery -> where(PostSaveTable::SAVED_TO_COLLECTION_ID, $collectionIdFromReq);
            }

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $postSaveTableQuery -> greaterThan(PostSaveTable::ID, $offset);
            }
            else
            {
                $postSaveTableQuery -> lessThan(PostSaveTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $postSaveTableQuery
                -> orderByDesc(PostSaveTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_POST_SAVE));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $postSaveBeans = $postSaveTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($postSaveBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | prepare post like maps
            |--------------------------------------------------------------------------
            */

            self::processBeans(PostSaveTable::getTableName(), $postSaveBeans, function (PostSaveModel $model)
            {
                /*
                |--------------------------------------------------------------------------
                | posts to fetch
                |--------------------------------------------------------------------------
                */

                self::addDependency(PostTable::getTableName(), $model -> getSavedPostId());

                /*
                |--------------------------------------------------------------------------
                | add PostSaveMode to additional as well because clients usually checks for
                | for additionals and reflect saved state according to that
                |--------------------------------------------------------------------------
                */

                self::addToAdditionalResponse(PostSaveTable::getTableName(), $model -> getDataMap());
            });

            /*
            |--------------------------------------------------------------------------
            | process dependencies first
            |--------------------------------------------------------------------------
            */

            self::processDependencies();

            /*
            |--------------------------------------------------------------------------
            | get dependency node for post table
            |--------------------------------------------------------------------------
            */

            $postsContainer = self::$dataDock -> getContainer(PostTable::getTableName());

            /*
            |--------------------------------------------------------------------------
            | if posts are fetched
            |--------------------------------------------------------------------------
            */

            if ($postsContainer -> isPopulated())
            {
                /*
                |--------------------------------------------------------------------------
                | add user ids to fetch
                |--------------------------------------------------------------------------
                */

                foreach ($postsContainer -> getDataMaps() as $postDataMap)
                {
                    self::addDependency(UserTable::getTableName(), $postDataMap[PostTable::getOwnerAttribute()]);
                }

                /*
                |--------------------------------------------------------------------------
                | additional data | post likes maps
                | -------------------------------------------------------------------------
                | help build like button
                |--------------------------------------------------------------------------
                */

                self::fetchModelsAndAddAsAdditional(
                    PostLikeTable::getTableName(),
                    array(
                        PostLikeTable::LIKED_POST_ID    => $postsContainer -> getIds(),
                        PostLikeTable::LIKED_BY_USER_ID => self::$authedUserModel -> getId(),
                    )
                );
            }
        });
    }
}
